import os
from flask import Flask, request, jsonify, url_for
from flask_cors import CORS
from werkzeug.utils import secure_filename
from ultralytics import YOLO

# Nom des dossiers de stockage
UPLOAD_FOLDER = 'uploads'
RESULTS_SUBFOLDER = 'results'
MODEL_PATH = 'model/best.pt'

def create_app():
    # Crée l’app Flask en configurant le dossier static
    app = Flask(
        __name__,
        static_folder='static',       # dossier local où se trouvent /uploads et /results
        static_url_path='/static'     # URL sous laquelle Flask sert ces fichiers
    )

    # Active CORS pour votre API et pour les fichiers statiques
    CORS(app, resources={
        r"/predict": {"origins": "*"},
        r"/static/*": {"origins": "*"}
    })

    # Charge le modèle YOLOv8 (seulement une fois)
    model = YOLO(MODEL_PATH)

    # Prépare les dossiers
    os.makedirs(UPLOAD_FOLDER, exist_ok=True)
    results_folder = os.path.join(app.static_folder, RESULTS_SUBFOLDER)
    os.makedirs(results_folder, exist_ok=True)

    @app.route('/')
    def home():
        return "Bienvenue sur l'API de prédiction de MQ‑WELD"

    @app.route('/predict', methods=['POST'])
    def predict():
        # Vérifie qu'on a bien reçu un fichier
        if 'file' not in request.files:
            return jsonify({'error': 'No file uploaded'}), 400

        # Sauvegarde du fichier uploadé
        file = request.files['file']
        filename = secure_filename(file.filename)
        input_path = os.path.join(UPLOAD_FOLDER, filename)
        file.save(input_path)

        # Prédiction et sauvegarde du résultat
        results = model(input_path)
        # .save(...) écrit dans le dossier courant : on passe results_folder en argument
        results[0].save(dirname=results_folder, exist_ok=True)

        # Construit l'URL publique de l'image résultat
        image_url = url_for(
            'static',
            filename=f'{RESULTS_SUBFOLDER}/{filename}',
            _external=True
        )

        return jsonify({'image_url': image_url})

    return app


